############# Quantitative Textanalyse mit dem Paket quanteda ##############
########### Document-feature- und Feature-co-ocurrence-Matrizen #############

# Zunächst und für alle Fälle: Arbeitsbereich komplett bereinigen (Housekeeping)
rm(list= ls(all= TRUE))

# Aufrufen einiger Pakete
library(quanteda)
library(quanteda.textplots)
library(quanteda.textstats)
library(quanteda.textmodels)
library(readtext)

# Die Texte in einem Unterordner mit readtext einlesen (txt-Format, UTF-8 codiert)
myCorpus <- readtext("CorpusKriegssammlung/*", encoding = "UTF-8")

# Das Korpus mit quanteda erstellen
ToxiCorpus <- corpus(myCorpus) 
class(ToxiCorpus)
# "corpus"    "character"


# Ein Tokens Objekt erstellen
# tokens() segmentiert Texte in einem Korpus nach Wortgrenzen
# in Token (Wörter oder Sätze).
# Standardmäßig segmentiert tokens() entlang von Trennzeichen 
# (in der Regel Leerzeichen)
ToxiCorpus_token <- tokens(ToxiCorpus)

# Zahlen und Satzzeichen entfernen
ToxiCorpus_token <- tokens(ToxiCorpus_token, remove_numbers = TRUE, remove_punct = TRUE)

# nun setzen wir noch alles in Kleinschreibung (lowercasing)
ToxiCorpus_token <- tokens_tolower(ToxiCorpus_token)

# und reinigen das Korpus noch ein wenig von OCR-Schrott
ToxiCorpus_token <-  tokens_remove(ToxiCorpus_token,
                                   pattern = c("^", "m", "s", "o", "b", "n", "h", "d", "f", "e", "w", "l", "k", 
                                                   "g", "v", "u", "a", "t","z", "c", "r", "j", "i", "x",
                                                   "co", "be", "de", "un", "en", "ge", "ver", "mk", "ii", "iii", "iv", "sic"),
                                   padding = TRUE) 


# Hier noch die Metadaten hinzufügen
docvars(ToxiCorpus_token, 'published') <- c("1904", "1898", "1907", "1894", "1907", 
                                      "1915", "1915", "1916", "1867", "1917", 
                                      "1916", "1919", "1914", "1918", "1917", 
                                      "1901", "1916", "1916", "1917", "1918", 
                                      "1915", "1917", "1918", "1915", "1914", 
                                      "1917", "1887", "1917", "1885", "1910") 

docvars(ToxiCorpus_token, 'category') <- c("Historische Drucke", "Historische Drucke", 
                                     "Historische Drucke", "Historische Drucke", 
                                     "Historische Drucke", "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke", "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke", "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918, Kinder- und Jugendbücher", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke", "Historische Drucke", 
                                     "Historische Drucke", 
                                     "Historische Drucke, Geschichte / Ethnographie / Geographie") 

docvars(ToxiCorpus_token, 'publisher') <- c("Sattler", "Walther", "Lehmann", "Hölder", 
                                      "Stalling", "Stilke", "Verlag der Weißen Bücher", 
                                      "Reimer", "Verlag des Missionshauses", "Mittler", 
                                      "Nister", "Hobbing", "Vieweg", "Stuttgart", 
                                      "Stilke", "Jansen", "Reichenbach", "Reiche", 
                                      "Frankenstein & Wagner", "Reichsdruckerei", 
                                      "Duncker", "Hofer", "Steinkopf", "Mutze", 
                                      "Weichert", "Süsserott", "Walther & Apolant", 
                                      "Druck und Verlag der Missionshandlung", 
                                      "Geographisches Institut", "Levysohn") 

docvars(ToxiCorpus_token, 'comment') <- c("P_Praeventiv", "P_Praeventiv", "P_Praeventiv", 
                                    "P_Praeventiv", "P_Praeventiv", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", "PDruckeAllg_bis1920", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeAllg_bis1920", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", "PDruckeAllg_bis1920", 
                                    "PDruckeAllg_bis1920", "PDruckeAllg_bis1920", 
                                    "PDruckeAllg_bis1920") 


head(docvars(ToxiCorpus_token)) # den Kopf der Metadaten betrachten


### Erstellen einer Document-Feature Matrix
# Die Funktion dfm() erstellt eine document-feature matrix (dfm) aus einem Tokens-Objekt.
dfmat_toxic <- dfm(ToxiCorpus_token)
print(dfmat_toxic) # hier sehen wir, dass absolute Zahlen verwendet werden
# (d.h. die Häufigkeiten der Vorkommen)
# Wie viele Features hat diese Matrix?

# Aus dieser Matrix können auch die Namen der Dokumente und features ("Wörter")
# ausgelesen werden, indem die Funktionen docnames() and featnames() verwendet werden.
head(docnames(dfmat_toxic), 20)
head(featnames(dfmat_toxic), 20)

# Da eine Matrix ausschließlich Werte einer Art - in diesem Fall Zahlen - enthält,
# können mit rowSums() und colSums() Randwerte berechnet werden.
head(rowSums(dfmat_toxic), 10)
head(colSums(dfmat_toxic), 10) 

# Was genau sieht man hier?

# Die häufigsten features können mit topfeatures() gefunden werden
topfeatures(dfmat_toxic, 10) # diese dfm enthält noch die Stopwords

# Oft ist es nicht sinnvoll, mit absoluten Zahlenwerten zu arbeiten.
# Die Funktion dfm_weight gewichtet die absoluten Feature-Häufigkeiten in einer dfm.
# Hier werden die Häufigkeiten über die Textlänge des Dokuments hinweg 
# normalisiert ('relative Häufigkeit': the proportion of the feature 
# counts of total feature counts)
dfmat_toxic_prop <- dfm_weight(dfmat_toxic, scheme  = "prop") 
print(dfmat_toxic_prop) # hier sind also die relativen Werte zu finden

# Jetzt entfernen wir die Stopwörter aus der Matrix mit der Funktion dfm_select()
# aus der zuerst erstellten Matrix
dfmat_toxic_nostop <- dfm_select(dfmat_toxic, pattern = stopwords("de"), 
                                 selection = "remove") 
print(dfmat_toxic_nostop)

# Je nachdem, welche Rechenoperationen man weiter durchführen will, gibt es über
# die Entfernung von Stopwords hinaus weitere Möglichkeiten der Matrixreduktion.
# Es können z.B. Features aufgrund ihrer Länge (Anzahl Zeichen) entfernt werden.
# Im Beispiel unten werden lediglich Features behalten, die aus mindestens fünf
# Zeichen bestehen
dfmat_toxic_long <- dfm_keep(dfmat_toxic_nostop, min_nchar = 5)
print(dfmat_toxic_long)

topfeatures(dfmat_toxic_long) # die 10 häufigsten Worte mit mindestens 5 Zeichen

# Oben wurde dfm_select() verwendet, das Features nach Mustern auswählt.
# dfm_trim() hingegen wählt Features nach Häufigkeiten aus.
# Mit der Funktion min_termfreq = 10 werden alle Features entfernt, die weniger
# als 10 mal im Korpus vorkommen.
dfmat_toxic_freq <- dfm_trim(dfmat_toxic, min_termfreq = 10)
print(dfmat_toxic_freq)

# Mit der Funktion max_docfreq = 0.1 werden features entfernt
# die in mehr als 10% der Dokumente verwendet werden.
dfmat_toxic_docfreq <- dfm_trim(dfmat_toxic, max_docfreq = 0.1, docfreq_type = "prop") 

print(dfmat_toxic_docfreq)

# Nun trimmen wir die Größe der Matrix im Hinblick auf term frequency 
# und doc frequency; in diesem Fall setzen wir die minimale Häufigkeit der Features 
# auf 10 Vorkommen, wobei die Features in mindestens 8 Dokumenten vorkommen sollen
# Die Stopwords hatten wir bereits zuvor entfernt
dfmat_toxic_trim <- dfm_trim(dfmat_toxic_nostop, min_termfreq = 10, min_docfreq = 8)
print(dfmat_toxic_trim)

#### Anwendungsfälle für dfms
### Relative Häufigkeiten von Begriffen, die in einem Diktionär vorkommen

# Nochmal das Diktionär mit verschiedenen Kategorien anlegen
dict <- dictionary(list(Rasse = c("bastard*", "blendl*", "halbblut", "mestiz*", 
                                  "mischrass*", "mulatte", "negr*"),
                        Sexualität = c("androgyn*", "dandy", "hermaphro*", 
                                       "homophil*", "homosex*", "masturb*", 
                                       "masochis*", "pädera*", "pervers*", "sadism*"),
                        Diskriminierung = c("affenmensch*", "aussätzig*", "barbar*", 
                                            "behind*", "kannib*", "invalid*", 
                                            "liliput*", "missgebild*", 
                                            "rückständ*", "wild*", "zwerg*"),
                        Ethnie = c("apache", "berber", "botokude", "eskimo", 
                                   "hottentott*", "indian*", "kaffer*", "kanak*", 
                                   "kreol*", "marron*", "cimarron*", "mohr", 
                                   "mongol*", "neger*", "pygmäe*", "zigeuner*")))

# Die im Diktionär enthaltenen Worte innerhalb des dfm nachschauen lassen
dfmat_toxic_dict <- dfm_lookup(dfmat_toxic_prop, dictionary = dict, levels = 1)
print(dfmat_toxic_dict)

# Die Matrix in eine Tabelle umwandeln
ToxiCorpus_dict_prop <- convert(dfmat_toxic_dict, to = "data.frame")
View(format(ToxiCorpus_dict_prop, scientific = FALSE)) # scientific = FALSE unterdrückt Eulersche Zahlen
# Nun lassen sich die relativen Anteile der im Diktionär enthaltenen Begriffe
# pro Text ablesen


#### Weiterer Anwendungsfall: Berechnen von Ähnlichkeiten zwischen Dokumenten
### Distanzmatrix
# textstat_dist() berechnet Matrizen von Distanzen zwischen Dokumenten 
# oder Features auf der Grundlage dieses dfm und gibt eine 
# sparse matrix der Abstände zurück; Standard ist die Euklidische Distanz,
# Alternativen sind "manhattan", "maximum", "canberra", "minkowski"
# Außerdem werden die Textlängen normalisiert
distance <- textstat_dist(dfm_weight(dfmat_toxic_trim, scheme = "prop")) 

# Was könnte das scheme = "prop" hier bedeuten?

# Ähnlichkeiten zwischen Texten als Cluster ausgeben
# Hierarchisches clustering wird auf dem 'distance'-Objekt ausgeführt
# as.dist() aus dem Paket "stats",
# hclust() aus dem Paket "stats", 
# hier agglomeratives hierarchisches Clustering mit Ward.D2
# Alternativen: "ward.D", "single", "complete", "average" etc.
pres_cluster <- hclust(as.dist(distance), method = "ward.D2")

# Dokumentnamen als Labels verwenden
pres_cluster$labels <- docnames(dfmat_toxic_trim)

# In drei Cluster unterteilen
clustering <- cutree(pres_cluster, 3)

# Ergebnis des Clusterings speichern
# output <- data.frame(clustering)
# write.csv(output, "Toxikorpus-Clustering-Euklid-WardD2.csv", row.names=T)

# Als Dendrogramm plotten und als Tiff-Datei ablegen
ppi <- 300
set.seed(1) 
tiff(paste0("30Werke-Kriegssammlung-Dendrogramm.tif"), width=14*ppi, height=8*ppi, compression = "lzw", res=ppi)
plot(pres_cluster, xlab = "Distanz", ylab = "Höhe", hang = -1, 
     main = "Ward.D2 Clustering von 30 Werken aus der Kriegssammlung\nEuklidische Distanz, basierend auf der normalisierten Tokenhäufigkeit")
rect.hclust(pres_cluster, 3, border = "red")
dev.off()

# Nun schauen wir uns an, welche Worte jeweils welches Cluster charakterisieren
corpusdfm.matrix <- as.matrix(dfmat_toxic_trim)

p_words <- colSums(corpusdfm.matrix) / sum(corpusdfm.matrix)

cluster_words <- lapply(unique(clustering), function(x){
  rows <- corpusdfm.matrix[ clustering == x , ]
  
  # alle Worte rausschmeißen, die nicht zum Cluster gehören
  rows <- rows[ , colSums(rows) > 0 ]
  
  colSums(rows) / sum(rows) - p_words[ colnames(rows) ]
})

# Der Code unten fasst die Cluster tabellarisch zusammen.
# Die Größe eines jeden Clusters und die wahrscheinlichsten 15 Worte 
# in jedem Cluster werden dargestellt.
# size gibt die Zahl der in jedem einzelnen Cluster enthaltenen Texte an
cluster_summary <- data.frame(cluster = unique(clustering),
                              size = as.numeric(table(clustering)), 
                              top_words = sapply(cluster_words, function(d){
                                paste(
                                  names(d)[ order(d, decreasing = TRUE) ][ 1:15 ], 
                                  collapse = ", ")
                              }),
                              stringsAsFactors = FALSE)

cluster_summary

#   cluster size
# 1       1   14
# 2       2   9
# 3       3   4
# 4       4   3

# top_words 
# 1 leute, herr, wasser, deutschland, bald, muß, hätte, wohl, 
# soldaten, mußte, ufer, feuer, volk, deutschen, franz 
# 2 sansibar, frauen, gesellschaft, wurden, unsere, deutsche, wurde, 
# neger, uhr, kinder, belgischen, gefangenen, essen, belgier, stets
# 3 jahrhundert, seit, handel, wurde, periode, wurden, jahrhunderts, abschnitt, 
# zeit, millionen, kam, städte, jahre, dr, handels
# 4   krieg, england, kriege, krieges, liebe, europäischen, rußland, ganz, eben,
# englischen, idee, frankreich, geistes, japan, europas

### Weitere Anwendung: Ähnlichkeiten zwischen Textpaaren
### Ähnlichkeitsmatrix auf dem gleichen proportionalen dfm analog zu oben
similarity <- textstat_simil(dfmat_toxic_prop, margin = "documents", 
                             method = "cosine")

as.list(similarity) # die Ähnlichkeitsmatrix mal anschauen

# dot plot der similarity für "Tropenkoller" anschauen
dotchart(as.list(similarity)$"PPN633196762.txt", xlab = "Kosinusähnlichkeit", pch = 19, main = "Tropenkoller. Ein Kolonial-Roman")
# Alternativen: "correlation", "cosine", "jaccard", "ejaccard", 
# "dice", "edice", "hamman", "simple matching"

# dot plot der similarity für "Auf den Missionsfeldern" anschauen
dotchart(as.list(similarity)$"PPN797023216.txt", xlab = "Kosinusähnlichkeit", pch = 19, main = "Auf den Missionsfeldern")

# Einen dot plots als Tiff-Datei ablegen
set.seed(1) 
tiff(paste0("30Werke-Toxikorpus-DotPlot-Tropenkoller.tif"), width=8*ppi, height=12*ppi, compression = "lzw", res=ppi)
dotchart(as.list(similarity)$"PPN633196762.txt", xlab = "Kosinusähnlichkeit", pch = 19, main = "Tropenkoller")
dev.off()

### Weitere Anwendungsmöglichkeit: Gruppierung von Dokumenten
# dfm_group() fasst Dokumente auf der Grundlage eines Vektors zusammen,
# der im Argument groups angegeben ist.
# Bei der Gruppierung von Dokumenten werden die Summen der
# Merkmalshäufigkeiten verwendet
# dfm_group() stellt also eine Filtermöglichkeit dar; lässt sich theoretisch auch zum
# Vergleich über verschiedene Zeiträume hinweg anwenden

docvars(dfmat_toxic)
# published / category / publisher / comment

dfmat_category <- dfm_group(dfmat_toxic, groups = category)
print(dfmat_category) # gibt hier dann absolute aufsummierte Zahlen 
# für nur noch 4 Dokumentkategorien aus


### Feature co-occurence matrix (fcm)

# Wenn ein Korpus groß ist, müssen Sie vor der Erstellung einer 
# featur co-occurence matrix die Merkmale der document-feature matrix auswählen. 
# Im folgenden Beispiel wurden zunächst alle Stoppwörter und Satzzeichen entfernt. 
# Dann werden nur Begriffe behalten, die mindestens 10 Mal vorkommen. 
dfmat_nostop_mintermfreq <- dfm_trim(dfmat_toxic_nostop, min_termfreq = 10)

topfeatures(dfmat_nostop_mintermfreq)

nfeat(dfmat_nostop_mintermfreq)
# 9718, eine überschaubar große Matrix

# Eine feature co-occurence matrix wird mit fcm() erstellt
fcmat_nostop_mintermfreq <- fcm(dfmat_nostop_mintermfreq)
dim(fcmat_nostop_mintermfreq)
# 9718 9718

# topfeatures() gibt die Worte aus, die am häufigsten miteinander vorkommen
topfeatures(fcmat_nostop_mintermfreq)

# Nun können die Features der feature co-occurence matrix ausgewählt werden,
# indem man fcm_select() benutzt
feat <- names(topfeatures(fcmat_nostop_mintermfreq, 50))
fcmat_nostop_mintermfreq_select <- fcm_select(fcmat_nostop_mintermfreq, pattern = feat, selection = "keep")
dim(fcmat_nostop_mintermfreq_select)
# 50 50

# Eine feature co-occurrence matrix kann dazu benutzt werden,
# um ein semantischs Netzwerk zu visualisieren; dazu benutzt man textplot_network().
size <- log(colSums(dfm_select(fcmat_nostop_mintermfreq_select, feat, selection = "keep")))

set.seed(144)
textplot_network(fcmat_nostop_mintermfreq_select, min_freq = 0.8, vertex_size = size / max(size) * 3)

